#!/usr/bin/env python3
"""
===============================================================================
 AnthroHeart Signed Release Packager
===============================================================================
Purpose:
  1. Signs your .7z file with your existing Christmas Identity Key.
  2. Generates a 'Verification Manifest' proving you are the author.
  3. Prepares a folder you can drag-and-drop to IPFS (Pinata) or Archive.org.

No crypto purchase required.
===============================================================================
"""

import os
import sys
import json
import hashlib
import shutil
from nacl.signing import SigningKey
from nacl.encoding import HexEncoder

# CONFIGURATION
FILE_PATH = "The_AnthroHeart_Collection_Bundle.7z"
IDENTITY_KEY = "anthroheart_chain/anthro_identity.key"
PUBLIC_KEY_FILE = "anthroheart_chain/anthro_public.key"
RELEASE_DIR = "AnthroHeart_Signed_Release"

def get_hash(filepath):
    print(f"⚙️  Hashing {filepath}...")
    sha256 = hashlib.sha256()
    sha512 = hashlib.sha512()
    
    total_size = os.path.getsize(filepath)
    processed = 0
    
    with open(filepath, "rb") as f:
        while chunk := f.read(1024 * 1024 * 16): # 16MB chunks
            sha256.update(chunk)
            sha512.update(chunk)
            processed += len(chunk)
            print(f"   Progress: {int((processed/total_size)*100)}%", end="\r")
            
    print("\n   Hashing complete.")
    return sha256.hexdigest(), sha512.hexdigest()

def main():
    # 1. SETUP
    if not os.path.exists(FILE_PATH):
        print(f"❌ Error: File not found: {FILE_PATH}")
        sys.exit(1)
        
    if not os.path.exists(IDENTITY_KEY):
        print(f"❌ Error: Identity key not found at {IDENTITY_KEY}")
        sys.exit(1)

    # Create release folder
    if os.path.exists(RELEASE_DIR):
        shutil.rmtree(RELEASE_DIR)
    os.makedirs(RELEASE_DIR)

    # 2. LOAD KEY
    print("🔑 Loading Identity Key...")
    with open(IDENTITY_KEY, "r") as f:
        private_hex = f.read().strip()
    signing_key = SigningKey(private_hex, encoder=HexEncoder)
    public_key_hex = signing_key.verify_key.encode(encoder=HexEncoder).decode()

    # 3. HASH & SIGN
    h256, h512 = get_hash(FILE_PATH)
    
    # Sign the SHA512 hash (most secure)
    signature = signing_key.sign(h512.encode()).signature.hex()

    # 4. CREATE MANIFEST
    manifest = {
        "title": "The AnthroHeart Collection",
        "file_name": FILE_PATH,
        "author_identity_public_key": public_key_hex,
        "hashes": {
            "sha256": h256,
            "sha512": h512
        },
        "signature_of_sha512": signature,
        "signature_algorithm": "Ed25519",
        "instructions": "To verify: Hash the 7z file (SHA512) and check against the signature using the public key."
    }

    # 5. PACKAGING
    print("📦 Packaging files...")
    
    # Move/Copy the big file
    dest_file = os.path.join(RELEASE_DIR, FILE_PATH)
    print(f"   Copying {FILE_PATH} to release folder (this might take a moment)...")
    shutil.copy2(FILE_PATH, dest_file)
    
    # Save Manifest
    with open(os.path.join(RELEASE_DIR, "manifest.json"), "w") as f:
        json.dump(manifest, f, indent=2)
        
    # Save Public Key for easy verification
    if os.path.exists(PUBLIC_KEY_FILE):
        shutil.copy2(PUBLIC_KEY_FILE, os.path.join(RELEASE_DIR, "author_public.key"))

    # Create a Human-Readable Readme
    readme_text = f"""
THE ANTHROHEART COLLECTION - SIGNED RELEASE
===========================================
Author: Thomas B. Sweet (Anthro Entertainment LLC)
Date:   2025-12-25

CRYPTOGRAPHIC PROOF:
--------------------
This archive is digitally signed.
Public Key: {public_key_hex}
SHA512 Hash: {h512}
Signature:   {signature}

This signature guarantees that the file "The_AnthroHeart_Collection_Bundle.7z" 
in this folder has not been altered since it was packaged by the author.
"""
    with open(os.path.join(RELEASE_DIR, "README_SIGNED.txt"), "w") as f:
        f.write(readme_text)

    print("\n" + "="*60)
    print(" READY FOR UPLOAD")
    print("="*60)
    print(f"📂 Created Folder:  {RELEASE_DIR}/")
    print(f"   Contains:        {FILE_PATH}")
    print(f"                    manifest.json")
    print(f"                    README_SIGNED.txt")
    print("-" * 60)
    print("HOW TO PUBLISH (Simple & Free):")
    print("1. Go to https://www.pinata.cloud (Create Free Account)")
    print("2. Click 'Add Files' -> 'Folder'")
    print(f"3. Select the '{RELEASE_DIR}' folder.")
    print("4. This puts your signed archive on IPFS (Decentralized & Permanent).")
    print("="*60)

if __name__ == "__main__":
    main()